#!/bin/env ruby
# encoding: utf-8
require 'rubygems'
require 'rspec'
require 'watir-webdriver'
require 'page-object'
require 'oci8'

require_relative '../../module/DriverUtility'
require_relative '../rspec_helper'
require_relative 'request_helper'
require_relative '../../pages/landing'
require_relative '../../pages/login'
require_relative '../../pages/eula'
require_relative '../../common/common'
require_relative '../../pages/landing_appointments'
require_relative '../../pages/appointments/new_appointment_request'
require_relative '../../pages/appointments/request_appointment_detail'
require_relative '../../helper/modal'
require_relative '../../module/database/OracleUtility'
require_relative '../../module/DateUtility'
require_relative '../../helper/error_validation'
require_relative '../../module/database/MongoUtility'

describe '[Story#VAR-4634: New Request - Type of Care: Express Care]
          [Story#VAR-4996: New Request - Express Care: Display facility supported end time]
          [Story#VAR-10481: Scheduling outside of express care hours - Error Message ]' do
  include OracleUtility
  include DateUtility
  include DriverUtility
  include MongoUtility

  before(:all) do
    initializeConfigurations(VAR_BASE_URL)
    @landing_appointments = LandingAppointments.new(@driver)
    @login = Login.new(@driver)
    @eula = Eula.new(@driver)
    @common = Common.new(@driver)
    @modal = Modal.new(@driver)
    @new_request = NewAppointmentRequest.new(@driver)
    @detail = RequestAppointmentDetail.new(@driver)
    @error = Error_Validation.new(@driver)

    #mongo
    @db = connectToMongoDB("var-utility")
    restoreCollection("siteSupportingVAR", "var-utility", "sites-supporting-var-exported2.json", @db)
    restoreCollection("clinical-services", "var-utility", "clinical_services_default.json", @db)
    restoreCollection("custom-friendly-text", "var-utility", "facility_friendly_name.json", @db)
    restoreCollection("requestEligibilityCriteria", "var-utility", "requestEligibilityCriteria.json", @db)
    #oracle
    @patientID = "1006088937V099668"
    @patientID2 = "1113138327"
    deletePatientData(@patientID)
    deletePatientData(@patientID2)

    @common.loginEndsOnHome(pw_redacted)

  end

  after(:all) do
    restoreCollection("requestEligibilityCriteria", "var-utility", "requestEligibilityCriteria.json", @db)
    @driver.close
  end

  context 'AC#VAR-4951|TC#VAR-4952: Test New Type of Care - Express Care
           AC#VAR-5194|TC#VAR-5195: Test - Express Care: Display facility supported end time
           AC#VAR-10639|TC#VAR-10640 : Scheduling outside of express care hours - Error Message ' do
    it "should verify Express Care is shown in Type of care list" do
      @common.selectAddAppointmentButton("VA")
      @new_request.type_of_care_element.when_visible(TIME_OUT_LIMIT)
      expect(@new_request.type_of_care_options).to eq(["Select", PRIMARY_CARE, AUDIOLOGY, EXPRESS_CARE , OPTOMETRY, MENTAL_HEALTH])
    end

    it "Should verify express care end time message" do
      @new_request.selectTypeOfCare(EXPRESS_CARE)
      @common.waitWhileSpinnerPresent()
      @new_request.selectState("523")
      @new_request.selectLocation("523")
      @common.waitWhileSpinnerPresent()
      expect(@new_request.express_care_end_time).to eq("Express Care request at this location must be submitted before 23:59 #{Time.now.zone}.")

    end

    it "Express Care is selected, the Scheduling Method is defaulted to 'Submit a Request to VA Scheduler'" do
      expect(@new_request.scheduling_method_radio_elements[1].value).to eq("clerk")
      expect(@new_request.scheduling_method_radio_elements[1].enabled?).to eq(true)
      expect(@new_request.scheduling_method_radio_elements[1].selected?).to eq(true)
      expect(@new_request.request_details_legend).to eq("Request Details - Express Care")
    end

    it "Schedule Myself option is disabled " do
      expect(@new_request.scheduling_method_radio_elements[0].value).to eq("direct")
      expect(@new_request.scheduling_method_radio_elements[0].enabled?).to eq(false)

    end

    it "should display a message is shown below the method that reads:  This method is not available for the type of care you selected." do
      expected_content =[\
                        "Step 4: Scheduling Method",
                        "The location you selected supports the following scheduling methods:",
                        "* Select a scheduling method:",
                        "Schedule Myself",
                        "This method is not available for the type of care you selected.",
                        "Submit a Request to VA Scheduler"
      ]

      actual_content = @new_request.scheduling_method_section.split("\n")
      compareContents(expected_content, actual_content)
    end

    it "Express Care enabled, but user does not have PACT" do
      @new_request.cancelForm
      @common.logoutAndLogbackWithADifferentUser(MockUser03)
      @common.selectAddAppointmentButton("VA")
      @new_request.selectTypeOfCare(EXPRESS_CARE)
      @common.waitWhileSpinnerPresent()
      @new_request.selectState("523")
      @new_request.selectLocation("523")
      expected_content =[\
                        "Step 4: Scheduling Method",
                        "The location you selected supports the following scheduling methods:",
                        "* Select a scheduling method:",
                        "Schedule Myself",
                        "This method is not available for the type of care you selected.",
                        "Submit a Request to VA Scheduler",
                        "You must have an assigned PACT in order to submit this type of request. Please contact your facility for assistance."
      ]

      actual_content = @new_request.scheduling_method_section.split("\n")
      compareContents(expected_content, actual_content)
    end

    it "Should verify both scheduling methods are disabled and none is selected" do

      expect(@new_request.scheduling_method_radio_elements[0].enabled?).to eq(false)
      expect(@new_request.scheduling_method_radio_elements[1].enabled?).to eq(false)
      expect(@new_request.scheduling_method_radio_elements[1].selected?).to eq(false)
      expect(@new_request.reason_for_visit_element.visible?).to eq(false)
    end

    it "If Express Care is not available, the location does not show up. Only available location with Express care appears" do
      restoreCollection("requestEligibilityCriteria", "var-utility", "requestEligibilityCriteriaNoExpressCare.json", @db)
      @new_request.cancelForm
      @landing_appointments.primary_header_element.when_present(TIME_OUT_LIMIT)
      @common.logoutAndLogbackWithADifferentUser(pw_redacted)
      @common.selectAddAppointmentButton("VA")
      @new_request.type_of_care_element.when_visible(TIME_OUT_LIMIT)
      @new_request.selectTypeOfCare(EXPRESS_CARE)
      @new_request.selectState("523")
      @new_request.location_instruction_element.wait_until_present(TIME_OUT_LIMIT)
      expect(@new_request.location_instruction_element.text).to eq "One location supports scheduling Express Care through this app. If you cannot find the location you need, please visit the VA Facility Locator to contact the facility for assistance."
      expect(@new_request.single_location).to eq "BROCKTON VAMC (BROCKTON, MA)"

    end

    it "should verify  TC#VAR-10640: Express Care is supported but the user has passed the can schedule end time" do
      restoreCollection("requestEligibilityCriteria", "var-utility", "requestEligibilityCriteriaTimePassedEC.json", @db)
      @new_request.cancelForm
      @landing_appointments.primary_header_element.when_present(TIME_OUT_LIMIT)
      @common.selectAddAppointmentButton("VA")
      @new_request.selectTypeOfCare(EXPRESS_CARE)
      @new_request.selectState("523")
      @new_request.selectLocation("523A5")
      expected_content =[\
                        "Step 4: Scheduling Method",
                        "The location you selected supports the following scheduling methods:",
                        "* Select a scheduling method:",
                        "Schedule Myself",
                        "This method is not available for the type of care you selected.",
                        "Submit a Request to VA Scheduler",
                        "This method is only available today between the hours of 02:00 and 03:00 #{Time.now.zone}"
      ]
      actual_content = @new_request.scheduling_method_section.split("\n")
      compareContents(expected_content, actual_content)
    end
  end
end







